/**
* \file: compositor_shim.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: compositor_shim
*
* \author: E. Ucan / ADITG/SW1 / eucan@de.adit-jv.com
*
* \copyright (c) 2015 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/time.h>

#include <wayland-client.h>
#include "WaylandServerinfoClientProtocol.h"
#ifndef WITH_WESTON
#include <ilm/ilm_client.h>
#include <ilm/ilm_control.h>
#endif
#include "compositor-shim.h"
#include "ivi-application-client-protocol.h"

struct compositor_shim_context
{
    struct wl_display* wlDisplay;
    struct wl_registry* wlRegistry;
    struct wl_event_queue* wlQueue;

    struct serverinfo* wlExtServerinfo;
    uint32_t connect_id;
    struct ivi_application* iviApp;
    struct wl_shell* wlShell;
};

static void serverinfoListener(void *data, struct serverinfo *pServerinfo, uint32_t client_handle)
{
    (void)pServerinfo;
    struct compositor_shim_context* p_wlCtx = (struct compositor_shim_context*)data;
    p_wlCtx->connect_id = client_handle;
    fprintf( stderr, "notified wayland connection : id=%d\n", p_wlCtx->connect_id);
}

struct serverinfo_listener serverinfo_listener_list = {
    serverinfoListener
};

static void registry_handle_global(void* data, struct wl_registry* registry, uint32_t name, const char* interface, uint32_t version)
{
    (void)version;
    struct compositor_shim_context* p_wlCtx = (struct compositor_shim_context*)data;
    int ans_strcmp;

    ans_strcmp = strcmp(interface, "serverinfo");
    if (0 == ans_strcmp)
    {
        p_wlCtx->wlExtServerinfo = (struct serverinfo*)wl_registry_bind(registry, name, &serverinfo_interface, 1);
        wl_proxy_set_queue((void*)p_wlCtx->wlExtServerinfo, p_wlCtx->wlQueue);
        serverinfo_add_listener(p_wlCtx->wlExtServerinfo, &serverinfo_listener_list, data);
        serverinfo_get_connection_id(p_wlCtx->wlExtServerinfo);
    }
    ans_strcmp = strcmp(interface, "ivi_application");
    if (0 == ans_strcmp)
    {
        p_wlCtx->iviApp = wl_registry_bind(registry, name, &ivi_application_interface, 1);
        wl_proxy_set_queue((void*)p_wlCtx->iviApp, p_wlCtx->wlQueue);
    }
    ans_strcmp = strcmp(interface, "wl_shell");
    if (0 == ans_strcmp)
    {
        p_wlCtx->wlShell = wl_registry_bind(registry, name, &wl_shell_interface, 1);
        wl_proxy_set_queue((void*)p_wlCtx->wlShell, p_wlCtx->wlQueue);
    }
}

static void registry_handle_global_remove(void* data, struct wl_registry* registry, uint32_t name)
{
    (void)data;
    (void)registry;
    (void)name;
}

static const struct wl_registry_listener registry_listener = {
    registry_handle_global,
    registry_handle_global_remove
};

/**
 * Destroys ilm_surface
 **
 *
 * @retval	>=0	if success.
 * @retval	<0	if error.
 */
int compositor_shim_surface_destroy(struct compositor_shim_surface_context* surf_ctx)
{
    if (surf_ctx == NULL)
    {
        fprintf(stderr, "Error: Surface context is not initialized\n");
        return -1;
    }

    if (surf_ctx->shim_ctx == NULL)
    {
        fprintf(stderr, "Error: Surface context does not have a compositor shim context\n");
        return -1;
    }

    if (surf_ctx->wlShellSurface)
    {
        wl_shell_surface_destroy(surf_ctx->wlShellSurface);
        surf_ctx->wlShellSurface = NULL;
    }

    if (surf_ctx->iviSurface)
    {
        ivi_surface_destroy(surf_ctx->iviSurface);
        surf_ctx->iviSurface = NULL;
    }
#ifndef WITH_WESTON
    ilmErrorTypes ilmError = ilm_surfaceRemove(surf_ctx->surface_id);

    if (ilmError == ILM_FAILED)
    {
        fprintf(stderr, "Error: ilm_surfaceRemove\n");
        return -1;
    }

    ilmError = ilm_commitChanges();

    if (ilmError == ILM_FAILED)
    {
        fprintf(stderr, "Error: ilm_commitChanges\n");
        return -1;
    }
#endif

    return 0;
}

/**
 * Terminates compositor_shim_context
 **
 *
 * @retval	>=0	if success.
 * @retval	<0	if error.
 */
int compositor_shim_terminate (struct compositor_shim_context* ctx)
{
    if (ctx == NULL)
    {
        fprintf(stderr, "Error: Compositor shim context is not initialized\n");
        return -1;
    }

    wl_display_flush(ctx->wlDisplay);
    wl_display_roundtrip_queue(ctx->wlDisplay, ctx->wlQueue);

    if (ctx->wlRegistry)
    {
        wl_registry_destroy(ctx->wlRegistry);
        ctx->wlRegistry = NULL;
    }

    if (ctx->wlQueue)
    {
        wl_event_queue_destroy(ctx->wlQueue);
        ctx->wlQueue = NULL;
    }

    if (ctx->wlExtServerinfo)
    {
        serverinfo_destroy(ctx->wlExtServerinfo);
        ctx->wlExtServerinfo = NULL;
    }

    if (ctx->iviApp)
    {
        ivi_application_destroy(ctx->iviApp);
        ctx->iviApp = NULL;
    }

    if (ctx->wlShell)
    {
        wl_shell_destroy(ctx->wlShell);
        ctx->wlShell = NULL;
    }

#ifndef WITH_WESTON
    ilmErrorTypes ilmError = ilm_commitChanges();
    if (ilmError == ILM_FAILED)
    {
        fprintf(stderr, "Error: ilm_commitChanges\n");
        return -1;
    }

    ilmError = ilm_destroy();
    if (ilmError == ILM_FAILED)
    {
        fprintf(stderr, "Error: ilm_destroy\n");
        return -1;
    }
#endif
    free(ctx);
    ctx = NULL;

    return 0;
}

/**
 * Create and Configure a ilm_surface
 **
 * For weston, the API just creates a surface.
 * For LayerManager, the API can set source/destination
 * rectangle, visibility and opacity
 *
 * @retval	>=0	if success.
 * @retval	<0	if error.
 */
int compositor_shim_surface_configure(struct compositor_shim_context* ctx,
                                      struct compositor_shim_surface_context* surf_ctx, uint32_t mask)
{
    if (ctx == NULL)
    {
        fprintf(stderr, "Error: Compositor shim context is not initialized\n");
        return -1;
    }

    if (surf_ctx == NULL)
    {
        fprintf(stderr, "Error: Layer context is not initialized\n");
        return -1;
    }

    surf_ctx->shim_ctx = ctx;

#ifdef WITH_WESTON
    if (mask & ADAPTER_CONFIGURATION_CREATE)
    {
        if (ctx->iviApp)
        {
            surf_ctx->iviSurface = ivi_application_surface_create(ctx->iviApp,
                                                                  surf_ctx->surface_id,
                                                                  surf_ctx->wlSurface);
        } else if (ctx->wlShell)
        {
            surf_ctx->wlShellSurface = wl_shell_get_shell_surface(ctx->wlShell, surf_ctx->wlSurface);

            if (surf_ctx->wlShellSurface)
                wl_shell_surface_set_toplevel(surf_ctx->wlShellSurface);
        }

        wl_display_flush(ctx->wlDisplay);
    }
#else
    struct wl_proxy* pxy = (struct wl_proxy*)surf_ctx->wlSurface;
    uint32_t id = (uint32_t) wl_proxy_get_id(pxy);
    uint32_t native_ilm_handle = (ctx->connect_id << 16) | id;
    ilmErrorTypes ilmError;

    if (mask & ADAPTER_CONFIGURATION_CREATE)
    {
        ilmError = ilm_surfaceCreate((t_ilm_nativehandle) native_ilm_handle, surf_ctx->origSourceWidth,
                                     surf_ctx->origSourceHeight, ILM_PIXELFORMAT_RGBA_8888,
                                     &surf_ctx->surface_id);

        if (ilmError == ILM_FAILED)
        {
            fprintf(stderr, "Error: ilm_surfaceCreate\n");
            return -1;
        }
    }

    if (mask & ADAPTER_CONFIGURATION_DEST_RECT)
    {
        ilmError = ilm_surfaceSetDestinationRectangle((t_ilm_surface)surf_ctx->surface_id, surf_ctx->destX,
                                                      surf_ctx->destY, surf_ctx->destWidth, surf_ctx->destHeight);
        if (ilmError == ILM_FAILED)
        {
            fprintf(stderr, "Error: ilm_surfaceSetDestinationRectangle\n");
            return -1;
        }
    }

    if (mask & ADAPTER_CONFIGURATION_SOURCE_RECT)
    {
        ilmError = ilm_surfaceSetSourceRectangle((t_ilm_surface)surf_ctx->surface_id, surf_ctx->sourceX,
                                                      surf_ctx->sourceY, surf_ctx->sourceWidth, surf_ctx->sourceHeight);
        if (ilmError == ILM_FAILED)
        {
            fprintf(stderr, "Error: ilm_surfaceSetSourceRectangle\n");
            return -1;
        }
    }

    if (mask & ADAPTER_CONFIGURATION_ADD)
    {
        ilmError = ilm_layerAddSurface((t_ilm_layer)surf_ctx->layer_id, (t_ilm_surface)surf_ctx->surface_id);
        if (ilmError == ILM_FAILED)
        {
            fprintf(stderr, "Error: ilm_layerAddSurface\n");
            return -1;
        }
    }


    if (mask & ADAPTER_CONFIGURATION_OPACITY)
    {
        ilmError = ilm_surfaceSetOpacity((t_ilm_surface)surf_ctx->surface_id, surf_ctx->opacity);
        if (ilmError == ILM_FAILED)
        {
            fprintf(stderr, "Error: ilm_surfaceSetOpacity\n");
            return -1;
        }
    }

    if (mask & ADAPTER_CONFIGURATION_VISIBILITY)
    {
        ilmError = ilm_surfaceSetVisibility((t_ilm_surface)surf_ctx->surface_id, surf_ctx->visibility);
        if (ilmError == ILM_FAILED)
        {
            fprintf(stderr, "Error: ilm_surfaceSetVisibility\n");
            return -1;
        }
    }

    ilmError = ilm_commitChanges();
    if (ilmError == ILM_FAILED)
    {
        fprintf(stderr, "fail 8\n");
        return -1;
    }
#endif
    return 0;
}

/**
 * Set destination/source rectangle of a surface with deferred API
 **
 * Deferred API ensures that the resize and buffer update happen in
 * one vsync. It does not work for weston.
 *
 * @retval	>=0	if success.
 * @retval	<0	if error.
 */
int compositor_shim_surface_configure_srcdst_deferred(struct compositor_shim_context* ctx,
                                      struct compositor_shim_surface_context* surf_ctx, uint32_t mask)
{
    if (ctx == NULL)
    {
        fprintf(stderr, "Error: Compositor shim context is not initialized\n");
        return -1;
    }

    if (surf_ctx == NULL)
    {
        fprintf(stderr, "Error: Layer context is not initialized\n");
        return -1;
    }

    surf_ctx->shim_ctx = ctx;

#ifdef WITH_WESTON
    (void)mask;
#else

    ilmErrorTypes ilmError;

    if (mask & ADAPTER_CONFIGURATION_SOURCE_RECT)
    {
        ilmError = ilm_surfaceSetSourceRectangleDeferred((t_ilm_surface)surf_ctx->surface_id, surf_ctx->sourceX,
                                                      surf_ctx->sourceY, surf_ctx->sourceWidth, surf_ctx->sourceHeight);
        if (ilmError == ILM_FAILED)
        {
            fprintf(stderr, "Error: ilm_surfaceSetSourceRectangle\n");
            return -1;
        }
    }

    if (mask & ADAPTER_CONFIGURATION_DEST_RECT)
    {
        ilmError = ilm_surfaceSetDestinationRectangleDeferred((t_ilm_surface)surf_ctx->surface_id, surf_ctx->destX,
                                                      surf_ctx->destY, surf_ctx->destWidth, surf_ctx->destHeight);
        if (ilmError == ILM_FAILED)
        {
            fprintf(stderr, "Error: ilm_surfaceSetDestinationRectangle\n");
            return -1;
        }
    }

    ilmError = ilm_commitChanges();
    if (ilmError == ILM_FAILED)
    {
        fprintf(stderr, "fail 8\n");
        return -1;
    }
#endif
    return 0;
}

/**
 * Initialize compositor_shim_context to default values
 **
 * This is a helper function to initialize compositor_shim_context
 *
 * @retval	>=0	if success.
 * @retval	<0	if error.
 */
int compositor_shim_surface_init(struct compositor_shim_surface_context* surf_ctx,
                                      struct wl_surface* wlSurface, uint32_t layer_id,
                                      uint32_t surface_id, uint32_t width, uint32_t height)
{
    if (wlSurface == NULL)
    {
        fprintf(stderr, "Error: wlSurface is not initialized\n");
        return -1;
    }

    if (surf_ctx == NULL)
    {
        fprintf(stderr, "Error: Surface context is not initialized\n");
        return -1;
    }

    surf_ctx->wlSurface = wlSurface;
    surf_ctx->wlShellSurface = NULL;
    surf_ctx->iviSurface = NULL;
    surf_ctx->surface_id = surface_id;
    surf_ctx->layer_id = layer_id;

    surf_ctx->origSourceWidth = width;
    surf_ctx->origSourceHeight = height;
    surf_ctx->sourceWidth = width;
    surf_ctx->sourceHeight = height;
    surf_ctx->destWidth = width;
    surf_ctx->destHeight = height;

    surf_ctx->sourceX = 0;
    surf_ctx->sourceY = 0;
    surf_ctx->destX = 0;
    surf_ctx->destY = 0;

    surf_ctx->visibility = 1;
    surf_ctx->opacity = 1.0f;

    return 0;
}

/**
 * Wait for LayerManager to initialize.
 **
 * This API blocks till LayerManager is initialized.
 * It does not work for weston.
 *
 * @retval	>=0	if success.
 * @retval	<0	if error.
 */
int compositor_shim_wait_server ()
{
    int ret = 0;

#ifndef WITH_WESTON
    ilmErrorTypes ilmError;
    ilmError = ilm_init();

    if (ilmError == ILM_FAILED)
    {
        fprintf(stderr, "Error: ilm_init\n");
        ret = -1;
    }
#endif

    return ret;
}

/**
 * Call ilm_destroy
 **
 * This API calls ilm_destroy.
 * The main difference from compositor_shim_terminate is
 * this API does not destroys compositor_shim context.
 * Call this if you called compositor_shim_wait_server before
 *
 * @retval	>=0	if success.
 * @retval	<0	if error.
 */
int compositor_shim_close_server ()
{
	int ret = 0;

#ifndef WITH_WESTON
    ilmErrorTypes ilmError;
    ilmError = ilm_destroy();

    if (ilmError == ILM_FAILED)
    {
        fprintf(stderr, "Error: ilm_destory\n");
        ret = -1;
    }
#endif

    return ret;
}

/**
 * Initialize ADIT compositor-shim API.
 **
 * This needs to be called before `wl_compositor_surface_create`
 *
 * @retval    Pointer to created compositor_shim_context if success.
 * @retval    NULL if error.
 */
struct compositor_shim_context* compositor_shim_initialize (struct wl_display* wlDisplay)
{
    struct compositor_shim_context* ctx;
#ifndef WITH_WESTON
    ilmErrorTypes ilmError;
#endif

    if (wlDisplay == NULL)
    {
        fprintf(stderr, "Error: wlDisplay is not initialized\n");
        goto err;
    }

    ctx = calloc(1, sizeof *ctx);
    if (ctx == NULL) {
        (void)wlDisplay;
        fprintf(stderr, "no memory to allocate wayland adapter context\n");
        goto err;
    }
    ctx->wlDisplay = wlDisplay;

    if (compositor_shim_wait_server () < 0) {
        goto err_context;
    }

    ctx->wlQueue = wl_display_create_queue(ctx->wlDisplay);
    if (ctx->wlQueue == NULL) {
        fprintf(stderr, "Error: Failed to get wlQueue\n");
        goto err_ilm;
    }

    ctx->wlRegistry = wl_display_get_registry(ctx->wlDisplay);
    if (ctx->wlRegistry == NULL) {
        wl_event_queue_destroy(ctx->wlQueue);
        fprintf(stderr, "Error: Failed to get wlRegistry\n");
        goto err_ilm;
    }

    wl_proxy_set_queue((void*)ctx->wlRegistry, ctx->wlQueue);
    wl_registry_add_listener(ctx->wlRegistry, &registry_listener, ctx);
    wl_display_roundtrip_queue(ctx->wlDisplay, ctx->wlQueue);
    /*Second roundtrip to get the connection id from LayerManager*/
    wl_display_roundtrip_queue(ctx->wlDisplay, ctx->wlQueue);

    return ctx;

err_ilm:
#ifndef WITH_WESTON
    ilmError = ilm_destroy();
    if (ilmError == ILM_FAILED)
    {
        fprintf(stderr, "Error: ilm_destroy\n");
    }
#endif
err_context:
    free(ctx);
err:
    return NULL;
}

